/**
 * This script renders a still image sequence from the current Vegas
 * project. It first presents a dialog that allows you to specify the
 * output directory, base file name, file format (PNG or JPEG), start
 * time, stop time, and step time.  Then it enters a loop that exports
 * a sequence of image files.  See below for more details.
 *
 * !! WARNING !! This script can take a very long time to complete.
 * IF YOU WISH TO STOP IT WHILE RUNNING, HIT THE <ESC> KEY.
 *
 * Revision Date: Aug. 01, 2005
 **/

using System;
using System.IO;
using System.Text;
using System.Drawing;
using System.Collections;
using System.Windows.Forms;
using ScriptPortal.Vegas;

public class EntryPoint
{
    
    public void FromVegas(Vegas vegas, String scriptPath)
    {

        // First save off the preview & project settings that will be
        // set by the script so they can be restored when the renders
        // are complete.
        VideoRenderQuality originalRenderQuality = vegas.Project.Preview.RenderQuality;
        VideoPreviewSize originalPreviewSize = vegas.Project.Preview.PreviewSize;
        VideoFieldOrder originalFieldOrder = vegas.Project.Video.FieldOrder;
        VideoDeinterlaceMethod originalDeinterlaceMethod = vegas.Project.Video.DeinterlaceMethod;
        Boolean needToRestoreSettings = false;

        Timecode originalCursor = vegas.Cursor;
        Timecode originalSelectionStart = vegas.SelectionStart;
        Timecode originalSelectionLength = vegas.SelectionLength;
        
        try {

            // Make a Timecode object whose value is zero for comparisons.
            Timecode zeroTime = new Timecode();

            // Set the default start, stop, and step times. If there is a
            // selection, use it for the start and stop times, otherwise do
            // the whole project.
            Timecode startTime, stopTime, stepTime;
            if (vegas.SelectionLength > zeroTime) {
                startTime = vegas.SelectionStart;
                stopTime = startTime + vegas.SelectionLength;
            } else {
                startTime = new Timecode(); // zero
                stopTime = vegas.Project.Length;
            }

            // The stepTime is the number of frames to skip forward between
            // each output file. The default value is 1 frame.
            stepTime = new Timecode();
            stepTime.FrameCount = 1;

            String projectName = "Untitled";
            if (null != vegas.Project.FilePath)
                projectName = Path.GetFileNameWithoutExtension(vegas.Project.FilePath);

            
            // Initialize list of file naming options.
            Object[] fileNamers = new Object[]
            {
                new FileNamer(),
                new BaseFileNamer("Image"),
                new BaseFileNamer(projectName),
                new TimecodeFileNamer(),
            };

            // Initialize list of file format options.
            Object[] fileFormats = new Object[]
            {
                new FileFormatItem(ImageFileFormat.JPEG, "JPEG", ".jpg"),
                new FileFormatItem(ImageFileFormat.PNG, "PNG", ".png"),
            };
            
            // Show the script's dialog box.
            RenderImageSequenceDialog dialog = new RenderImageSequenceDialog(vegas);
            if (null != vegas.Project.FilePath)
                dialog.OutputDirectory = Path.GetDirectoryName(vegas.Project.FilePath);
            else
                dialog.OutputDirectory = Path.GetPathRoot(scriptPath);
            dialog.StartTime = startTime;
            dialog.StopTime = stopTime;
            dialog.StepTime = stepTime;
            dialog.FileNamers = fileNamers;
            dialog.FileFormats = fileFormats;
            DialogResult dialogResult = dialog.ShowDialog();

            // if the OK button was not pressed, just return
            if (System.Windows.Forms.DialogResult.OK != dialogResult)
                return;

            // Get the basis for output image file names
            String outputDir = dialog.OutputDirectory;
            if (!Directory.Exists(outputDir))
                throw new ApplicationException("output directory does not exist");

            // Get the output image file name extension and corresponding
            // file format. ImageFileFormat.None indicates that the
            // current prefs setting will be used but that may not
            // correspond to the specified file extension.

            //String imageFileNameExt = Path.GetExtension(imageFileName);
            String fileExt = dialog.FileFormat.Extension;
            ImageFileFormat imageFormat = dialog.FileFormat.Format;
                
            // Get the start, stop, and step times specified in the
            // dialog.
            startTime = dialog.StartTime;
            stopTime = dialog.StopTime;
            stepTime = dialog.StepTime;

            // Make sure the step time is legal.
            if (stepTime <= zeroTime) {
                throw new ApplicationException("step time must be greater than zero");
            }

            // compute the total number of image files that will
            // be rendered.
            Timecode deltaTime = stopTime - startTime;
            if (zeroTime > deltaTime) {
                throw new ApplicationException("start time must be before the stop time");
            }
            long deltaFrames = deltaTime.FrameCount;
            long stepFrames = stepTime.FrameCount;
            if (0 == stepFrames) {
                throw new ApplicationException("insufficient step time");
            }
            int fileCount = (int) (deltaFrames / stepFrames);

            // prepare the file namer.
            FileNamer fileNamer = dialog.FileNamer;
            fileNamer.FileCount = fileCount;
            fileNamer.StartTime = startTime;
            fileNamer.StopTime = stopTime;
            fileNamer.StepTime = stepTime;
                
            // compute the list of image files that will be rendered.
            ArrayList snapshots = new ArrayList();
            fileNamer.FileIndex = 0;
            fileNamer.CurrentTime = Timecode.FromNanos(startTime.Nanos);
            DialogResult overwritePrompt = DialogResult.None;
            
            while (fileNamer.CurrentTime <= stopTime) {

                // compose the file name
                String filename = Path.Combine(outputDir, fileNamer.NameFile() + fileExt);
                bool fileExists = File.Exists(filename);

                // if the file already exists, prompt the user
                // (only once).
                if (fileExists && (DialogResult.None == overwritePrompt)) {
                    String msg = "One or more image files already exist.  Do you want to overwrite them?";
                    String cap = "WARNING: Files Exist";
                    MessageBoxButtons btns = MessageBoxButtons.YesNoCancel;
                    MessageBoxIcon icon = MessageBoxIcon.Warning;
                    MessageBoxDefaultButton defBtn = MessageBoxDefaultButton.Button3;
                    overwritePrompt = MessageBox.Show(msg, cap, btns, icon, defBtn);
                }

                if (DialogResult.Cancel == overwritePrompt) {
                    // skip all images
                    snapshots.Clear();
                    break;
                } else if (fileExists && (DialogResult.No == overwritePrompt)) {
                    // skip this one but allow others
                } else {
                    snapshots.Add(new SnapshotInfo(filename, Timecode.FromNanos(fileNamer.CurrentTime.Nanos)));
                }

                // increment the image index and current time.
                fileNamer.FileIndex++;
                fileNamer.CurrentTime += stepTime;
            }

            if (0 < snapshots.Count) {
                needToRestoreSettings = true;

                // Seek to the start frame
                vegas.Cursor = startTime;

                // Set the preview quality and size.
                vegas.Project.Preview.RenderQuality = VideoRenderQuality.Best;
                vegas.Project.Preview.PreviewSize = VideoPreviewSize.Full;

                // Set the field order and deinterlace method
                vegas.Project.Video.FieldOrder = VideoFieldOrder.ProgressiveScan;
                vegas.Project.Video.DeinterlaceMethod = VideoDeinterlaceMethod.InterpolateFields;

                // render the snapshots.
                foreach (SnapshotInfo snapshot in snapshots) {
                    RenderStatus renderStatus = vegas.SaveSnapshot(snapshot.File, imageFormat, snapshot.Time);
                    if (RenderStatus.Complete != renderStatus) {
                        break;
                    }
                }
            }

        } catch (Exception e) {
            MessageBox.Show(e.Message);

        } finally {

            // if needed, restore the project and preview settings
            if (needToRestoreSettings) {
                vegas.Project.Preview.RenderQuality = originalRenderQuality;
                vegas.Project.Preview.PreviewSize = originalPreviewSize;
                vegas.Project.Video.FieldOrder = originalFieldOrder;
                vegas.Project.Video.DeinterlaceMethod = originalDeinterlaceMethod;
                vegas.Cursor = originalCursor;
                vegas.SelectionStart = originalSelectionStart;
                vegas.SelectionLength = originalSelectionLength;
            }

        }
    }
}



// Form subclass that is the dialog box for this script
public class RenderImageSequenceDialog : Form
{
    private Vegas myVegas;

    private TextBox myOutputDirBox = new TextBox();
    private Button myBrowseButton = new Button();

    private ComboBox myFileNamerCombo = new ComboBox();
    private ComboBox myFileFormatCombo = new ComboBox();
    
    private TextBox myStartTimeBox = new TextBox();
    private TextBox myStopTimeBox = new TextBox();
    private TextBox myStepTimeBox = new TextBox();

    private Button myOKButton = new Button();
    private Button myCancelButton = new Button();
    
    public RenderImageSequenceDialog(Vegas vegas)
    {
        myVegas = vegas;
        
        this.Text = "Render Image Sequence";
        this.FormBorderStyle = System.Windows.Forms.FormBorderStyle.FixedDialog;
        this.MaximizeBox = false;
        this.StartPosition = FormStartPosition.CenterScreen;
        this.Width = 540;
        
        int leftMargin = 4;
        int topMargin = 8;
        int buttonWidth = 80;
        int ffWidth = 420;
        int fflWidth = 120;
        int top = (2 * topMargin);
        
        AddLabeledControl("Output Directory:", leftMargin, top, fflWidth, ffWidth, myOutputDirBox);

        myBrowseButton.Location = new Point(myOutputDirBox.Right + 4, myOutputDirBox.Top - 2);
        myBrowseButton.Width = buttonWidth;
        myBrowseButton.Text = "Browse...";
        myBrowseButton.FlatStyle = FlatStyle.System;
        myBrowseButton.Click += new EventHandler(this.HandleBrowseButtonClick);
        Controls.Add(myBrowseButton);

        top = myBrowseButton.Bottom + topMargin;

        myFileNamerCombo.DropDownStyle = ComboBoxStyle.DropDownList;
        AddLabeledControl("File Names:", leftMargin, top, fflWidth, ffWidth, myFileNamerCombo);

        myFileFormatCombo.DropDownStyle = ComboBoxStyle.DropDownList;
        myFileFormatCombo.Location = new Point(myFileNamerCombo.Right + leftMargin, top);
        myFileFormatCombo.Width = 86;
        Controls.Add(myFileFormatCombo);

        int tclWidth = 120;
        int tcWidth = 200;
        top = myFileNamerCombo.Bottom + topMargin;
        AddLabeledControl("Start Time:", leftMargin, top, tclWidth, tcWidth, myStartTimeBox);
        top = myStartTimeBox.Bottom + topMargin;
        AddLabeledControl("Stop Time:", leftMargin, top, tclWidth, tcWidth, myStopTimeBox);
        top = myStopTimeBox.Bottom + topMargin;
        AddLabeledControl("Step Time:", leftMargin, top, tclWidth, tcWidth, myStepTimeBox);
        
        top = myStepTimeBox.Bottom + (2 * topMargin);

        myOKButton.Text = "OK";
        myOKButton.FlatStyle = FlatStyle.System;
        myOKButton.Location = new Point(this.Width - (2*(buttonWidth+10)), top);
        myOKButton.Width = buttonWidth;
        myOKButton.DialogResult = DialogResult.OK;
        AcceptButton = myOKButton;
        Controls.Add(myOKButton);

        myCancelButton.Text = "Cancel";
        myCancelButton.FlatStyle = FlatStyle.System;
        myCancelButton.Location = new Point(this.Width - (1*(buttonWidth+10)), top);
        myCancelButton.Width = buttonWidth;
        myCancelButton.DialogResult = DialogResult.Cancel;
        CancelButton = myCancelButton;
        Controls.Add(myCancelButton);

        this.ClientSize = new Size(this.ClientSize.Width, myOKButton.Bottom + 8);
    }
    
    void AddLabeledControl(String labelText, int left, int top, int labelWidth, int totalWidth, Control control)
    {
        Label label = new Label();
        label.Text = labelText;
        label.Location = new Point(left, top);
        label.Width = labelWidth;
        label.TextAlign = ContentAlignment.MiddleLeft;
        Controls.Add(label);
        
        control.Location = new Point(label.Right, top);
        control.Width = totalWidth - labelWidth;
        Controls.Add(control);

    }

    public String OutputDirectory
    {
        get { return myOutputDirBox.Text; }
        set { myOutputDirBox.Text = value; }
    }
    
    public Timecode StartTime
    {
        get { return Timecode.FromString(myStartTimeBox.Text); }
        set { myStartTimeBox.Text = value.ToString(); }
    }

    public Timecode StopTime
    {
        get { return Timecode.FromString(myStopTimeBox.Text); }
        set { myStopTimeBox.Text = value.ToString(); }
    }

    public Timecode StepTime
    {
        get { return Timecode.FromString(myStepTimeBox.Text); }
        set { myStepTimeBox.Text = value.ToString(); }
    }

    public Object[] FileNamers
    {
        set
        {
            myFileNamerCombo.Items.Clear();
            myFileNamerCombo.Items.AddRange(value);
            myFileNamerCombo.SelectedIndex = 0;
        }
    }
    
    public FileNamer FileNamer
    {
        get { return (FileNamer) myFileNamerCombo.SelectedItem; }
        set { myFileNamerCombo.SelectedItem = value; }
    }

    public Object[] FileFormats
    {
        set
        {
            myFileFormatCombo.Items.Clear();
            myFileFormatCombo.Items.AddRange(value);
            myFileFormatCombo.SelectedIndex = 0;
        }
    }

    public FileFormatItem FileFormat
    {
        get { return (FileFormatItem) myFileFormatCombo.SelectedItem; }
        set { myFileFormatCombo.SelectedItem = value; }
    }
    
    protected void HandleBrowseButtonClick(Object sender, EventArgs e)
    {
        String outputDir = null;
        myVegas.FileUtilities.SelectDirectoryDlg(this.Handle,
                                                 "Select Output Directory",
                                                 myOutputDirBox.Text,
                                                 false,
                                                 out outputDir);
        if (null == outputDir)
            return; // canceled
        OutputDirectory = outputDir;
    }

}

public class SnapshotInfo
{
    public String File;
    public Timecode Time;
    
    public SnapshotInfo(String file, Timecode time)
    {
        this.File = file;
        this.Time = time;
    }
}

public class FileFormatItem
{
    public ImageFileFormat Format;
    public String Name;
    public String Extension;

    public FileFormatItem(ImageFileFormat fmt, String name, String ext)
    {
        this.Format = fmt;
        this.Name = name;
        this.Extension = ext;
    }

    public override String ToString()
    {
        return String.Format("{0} ({1})", this.Name, this.Extension);
    }
}

public class FileNamer
{

    // The following variables can be used by NameFile.
    public String PorjectName;
    public int FileIndex;
    public int FileCount;
    public Timecode StartTime;
    public Timecode StopTime;
    public Timecode StepTime;
    public Timecode CurrentTime;
    
    public virtual String NameFile()
    {
        int renderCountDigits = this.FileCount.ToString().Length;
        return String.Format("{0:D" + renderCountDigits + "}", this.FileIndex);
    }


    public override String ToString()
    {
        this.FileCount = 100;
        this.StartTime = Timecode.FromNanos(0);
        this.StopTime = Timecode.FromFrames(100);
        this.StepTime = Timecode.FromFrames(1);

        this.CurrentTime = Timecode.FromFrames(0);
        this.FileIndex = 0;
        String n0 = NameFile();

        this.CurrentTime = Timecode.FromFrames(1);
        this.FileIndex = 1;
        String n1 = NameFile();

        this.CurrentTime = Timecode.FromFrames(2);
        this.FileIndex = 2;
        String n2 = NameFile();

        return String.Format("{0}, {1}, {2}, ...", n0, n1, n2);
    }

}

public class BaseFileNamer : FileNamer
{
    public String BaseName;
    
    public BaseFileNamer(String baseName)
    {
        this.BaseName = baseName;
    }

    public override String NameFile()
    {
        return String.Format("{0}_{1}", this.BaseName, base.NameFile());
    }
}

public class TimecodeFileNamer : FileNamer
{
    public override String NameFile()
    {
        StringBuilder name = new StringBuilder(this.CurrentTime.ToString());
        name.Replace(':', '_');
        name.Replace(';', '_');
        name.Replace('.', '_');
        name.Replace(',', '_');
        name.Replace('+', '_');
        return name.ToString();
    }
}
